
import java.io.FileReader

/**
 * The main class of the CommentStrip utility.
 *
 * This program implements the following finite state machine:
 *
 * {{{
 * Normal -- '/'    --> MaybeComment
 * Normal -- '"'    --> DoubleQuote (print character)
 * Normal -- '\''   --> SingleQuote (print character)
 * Normal -- others --> Normal (print character)
 *
 * MaybeComment -- '/'    --> SlashSlashComment
 * MaybeComment -- '*'    --> BlockComment (print a space)
 * MaybeComment -- '+'    --> NestedComment (print a space; set count to 1)
 * MaybeComment -- '"'    --> DoubleQuote (print slash; print character)
 * MaybeComment -- '\''   --> SingleQuote (print slash; print character)
 * MaybeComment -- others --> Normal (print slash; print character)
 *
 * SlashSlashComment -- '\n'   --> Normal (print '\n')
 * SlashSlashComment -- others --> SlashSlashComment
 *
 * BlockComment -- '*'    --> MaybeUncomment
 * BlockComment -- '\n'   --> BlockComment (print character)
 * BlockComment -- others --> BlockComment
 *
 * MaybeUncomment -- '/'    --> Normal
 * MaybeUncomment -- '*'    --> MaybeUncomment
 * MaybeUncomment -- others --> BlockComment
 *
 * NestedComment -- '/'    --> MaybeNestedComment
 * NestedComment -- '+'    --> MaybeNestedUncomment
 * NestedComment -- '\n'   --> NestedComment (print character)
 * NestedComment -- others --> NestedComment
 *
 * MaybeNestedComment -- '+'    --> NestedComment (inc count)
 * MaybeNestedComment -- '/'    --> MaybeNestedComment
 * MaybeNestedComment -- others --> NestedComment
 *
 * MaybeNestedUncomment -- '/'    --> (dec count; if (count == 0) Normal else NestedComment
 * MaybeNestedUncomment -- '+'    --> MaybeNestedUncomment
 * MaybeNestedUncomment -- others --> NestedComment
 *
 * DoubleQuote -- '\\'   --> EscapeOneDouble (print character)
 * DoubleQuote -- '"'    --> Normal (print character)
 * DoubleQuote -- others --> DoubleQuote (print character)
 *
 * SingleQuote -- '\\'   --> EscapeOneSingle (print character)
 * SingleQuote -- '\''   --> Normal (print character)
 * SingleQuote -- others --> SingleQuote (print character)
 *
 * EscapeOneDouble -- others --> DoubleQuote (print character)
 * EscapeOneSingle -- others --> SingleQuote (print character)
 * }}}
 */
object CommentStrip:

  enum StateType:
    case Normal
    case MaybeComment
    case SlashSlashComment
    case BlockComment
    case MaybeUncomment
    case NestedComment
    case MaybeNestedComment
    case MaybeNestedUncomment
    case DoubleQuote
    case SingleQuote
    case EscapeOneDouble
    case EscapeOneSingle
  end StateType

  private def print(ch: Int): Unit =
    System.out.print(ch.asInstanceOf[Char])

  def main(args: Array[String]): Unit =
    var state = StateType.Normal
    val input = new FileReader(args(0))
    var count         = 0  // The nesting depth of nesting comments.
    var lineNumber    = 1  // The current line.
    var columnNumber  = 0  // The current column.
    var commentLine   = 0  // The line coordinate of the last block comment opening.
    var commentColumn = 0  // The column coordinate of the last block comment opening.
    var ch: Int       = 0  // A character from the input file.

    while { ch = input.read(); ch != -1 } do
      // Adjust file coordinates.
      if ch == '\n' then
        lineNumber  += 1
        columnNumber = 0
      else
        columnNumber += 1

      // Process this character.
      state match
        case StateType.Normal =>
          ch match
            case '/'  =>            state = StateType.MaybeComment
            case '"'  => print(ch); state = StateType.DoubleQuote
            case '\'' => print(ch); state = StateType.SingleQuote
            case _    => print(ch)

        case StateType.MaybeComment =>
          ch match
            case '/'  => state = StateType.SlashSlashComment

            case '*'  =>
              print(' ')
              state = StateType.BlockComment
              commentLine = lineNumber
              commentColumn = columnNumber - 1

            case '+'  =>
              print(' ')
              count = 1
              state = StateType.NestedComment
              commentLine = lineNumber
              commentColumn = columnNumber - 1

            case '"'  => print('/'); print(ch); state = StateType.DoubleQuote
            case '\'' => print('/'); print(ch); state = StateType.SingleQuote
            case _    => print('/'); print(ch); state = StateType.Normal
          end match

        case StateType.SlashSlashComment =>
          if ch == '\n' then
            print('\n')
            state = StateType.Normal

        case StateType.BlockComment =>
          ch match
            case '*'  =>            state = StateType.MaybeUncomment
            case '\n' => print(ch)
            case _    =>

        case StateType.MaybeUncomment =>
          ch match
            case '/'  => state = StateType.Normal
            case '*'  =>
            case _    => state = StateType.BlockComment

        case StateType.NestedComment =>
          ch match
            case '/'  =>            state = StateType.MaybeNestedComment
            case '+'  =>            state = StateType.MaybeNestedUncomment
            case '\n' => print(ch)
            case _    =>

        case StateType.MaybeNestedComment =>
          ch match
            case '+' => count += 1; state = StateType.NestedComment
            case '/' =>
            case _   =>             state = StateType.NestedComment

        case StateType.MaybeNestedUncomment =>
          ch match
            case '/' =>
              count -= 1
              if (count == 0) state = StateType.Normal else state = StateType.NestedComment
            case '+' =>
            case _   => state = StateType.NestedComment

        case StateType.DoubleQuote =>
          ch match
            case '\\' => print(ch); state = StateType.EscapeOneDouble
            case '"'  => print(ch); state = StateType.Normal
            case _    => print(ch)

        case StateType.SingleQuote =>
          ch match
            case '\\' => print(ch); state = StateType.EscapeOneSingle
            case '\'' => print(ch); state = StateType.Normal
            case _    => print(ch)

        case StateType.EscapeOneDouble =>
          print(ch)
          state = StateType.DoubleQuote

        case StateType.EscapeOneSingle =>
          print(ch)
          state = StateType.SingleQuote
      end match
    end while

    // Check the state at the end of the input.
    state match
      case StateType.BlockComment |
           StateType.MaybeUncomment |
           StateType.NestedComment |
           StateType.MaybeNestedComment |
           StateType.MaybeNestedUncomment =>
        printf("Unclosed block comment at end of input. Comment starts here: (%d, %d)", commentLine, commentColumn)

      case StateType.DoubleQuote |
           StateType.EscapeOneDouble =>
        printf("Unclosed double quoted string at end of input")

      case StateType.SingleQuote |
           StateType.EscapeOneSingle =>
        printf("Unclosed multi-character literal at end of input")

      case _ =>
        // No error.
    end match

    input.close()
  end main
end CommentStrip
