
/**
 * The main class of the CommentStripFunctional utility.
 *
 * This program implements the following finite state machine:
 *
 * {{{
 * Normal -- '/'    --> MaybeComment
 * Normal -- '"'    --> DoubleQuote (print character)
 * Normal -- '\''   --> SingleQuote (print character)
 * Normal -- others --> Normal (print character)
 *
 * MaybeComment -- '/'    --> SlashSlashComment
 * MaybeComment -- '*'    --> BlockComment (print a space)
 * MaybeComment -- '"'    --> DoubleQuote (print slash; print character)
 * MaybeComment -- '\''   --> SingleQuote (print slash; print character)
 * MaybeComment -- others --> Normal (print slash; print character)
 *
 * SlashSlashComment -- '\n'   --> Normal (print '\n')
 * SlashSlashComment -- others --> SlashSlashComment
 *
 * BlockComment -- '*'    --> MaybeUncomment
 * BlockComment -- '\n'   --> BlockComment (print character)
 * BlockComment -- others --> BlockComment
 *
 * MaybeUncomment -- '/'    --> Normal
 * MaybeUncomment -- '*'    --> MaybeUncomment
 * MaybeUncomment -- others --> BlockComment
 *
 * DoubleQuote -- '\\'   --> EscapeOneDouble (print character)
 * DoubleQuote -- '"'    --> Normal (print character)
 * DoubleQuote -- others --> DoubleQuote (print character)
 *
 * SingleQuote -- '\\'   --> EscapeOneSingle (print character)
 * SingleQuote -- '\''   --> Normal (print character)
 * SingleQuote -- others --> SingleQuote (print character)
 *
 * EscapeOneDouble -- others --> DoubleQuote (print character)
 * EscapeOneSingle -- others --> SingleQuote (print character)
 * }}}
 */
object CommentStripFunctional {

  enum StateType:
    case Normal
    case MaybeComment
    case SlashSlashComment
    case BlockComment
    case MaybeUncomment
    case DoubleQuote
    case SingleQuote
    case EscapeOneDouble
    case EscapeOneSingle

  private type DFAState = (StateType, List[Char])

  private def combiner(accumulator: DFAState, ch: Char): DFAState =
    val (state, currentOutput) = accumulator

    state match
      case StateType.Normal =>
        ch match
          case '/'  => (StateType.MaybeComment, currentOutput)
          case '"'  => (StateType.DoubleQuote, ch :: currentOutput)
          case '\'' => (StateType.SingleQuote, ch :: currentOutput)
          case _    => (state, ch :: currentOutput)

      case StateType.MaybeComment =>
        ch match
          case '/' => (StateType.SlashSlashComment, currentOutput)
          case '*' => (StateType.BlockComment, ' ' :: currentOutput)
          case '"' => (StateType.DoubleQuote, ch :: '/' :: currentOutput)
          case '\''=> (StateType.SingleQuote, ch :: '/' :: currentOutput)
          case _   => (StateType.Normal, ch :: '/' :: currentOutput)

      case StateType.SlashSlashComment =>
        if ch == '\n' then
          (StateType.Normal, '\n' :: currentOutput)
        else
          (state, currentOutput)

      case StateType.BlockComment =>
        ch match
          case '*'  => (StateType.MaybeUncomment, currentOutput)
          case '\n' => (state, ch :: currentOutput)
          case _    => (state, currentOutput)

      case StateType.MaybeUncomment =>
        ch match
          case '/' => (StateType.Normal, currentOutput)
          case '*' => (state, currentOutput)
          case _   => (StateType.BlockComment, currentOutput)

      case StateType.DoubleQuote =>
        ch match
          case '\\' => (StateType.EscapeOneDouble, ch :: currentOutput)
          case '"'  => (StateType.Normal, ch :: currentOutput)
          case _    => (state, ch :: currentOutput)

      case StateType.SingleQuote =>
        ch match
          case '\\' => (StateType.EscapeOneSingle, ch :: currentOutput)
          case '\'' => (StateType.Normal, ch :: currentOutput)
          case _    => (state, ch :: currentOutput)

      case StateType.EscapeOneDouble =>
        (StateType.DoubleQuote, ch :: currentOutput)

      case StateType.EscapeOneSingle =>
        (StateType.SingleQuote, ch :: currentOutput)
    end match

  end combiner

  def main(args: Array[String]): Unit = {
    val text = io.Source.fromFile(args(0))
    val (_, output) = text.foldLeft(StateType.Normal, List[Char]())(combiner)
    output.reverse foreach { System.out.print }
  }

}
